package org.monazilla.v2c;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.bouncycastle.tls.CertificateRequest;
import org.bouncycastle.tls.TlsAuthentication;
import org.bouncycastle.tls.TlsCredentials;
import org.bouncycastle.tls.TlsServerCertificate;
import org.bouncycastle.tls.crypto.TlsCertificate;

/**
 * TLS socket factory powered by bouncyCastle
 * @author koji.hayakawa
 * https://qiita.com/a__i__r/items/b75a381bf46a863b1139
 * https://github.com/a--i--r/TLSSocketFactory
 *
 */
public class V2CTLSAuthentication implements TlsAuthentication {

	protected V2CTLSClient tlsClient;
	protected static KeyStore keyStore = null;

	private static final Pattern patternVersion = Pattern
			.compile(".+?Version:\\s([^\n\r]+).+", Pattern.DOTALL);
	private static final Pattern patternSubject = Pattern
			.compile(".+?Subject:\\s([^\n\r]+).+", Pattern.DOTALL);
	private static final Pattern patternValidity = Pattern
			.compile(".+?Validity:\\s(\\[[^\\]]+\\]).+", Pattern.DOTALL);
	private static final Pattern patternIssuer = Pattern
			.compile(".+?Issuer:\\s([^\n\r]+).+", Pattern.DOTALL);
	private static HashMap hmCertificateExpiredException = null;
	
	public V2CTLSAuthentication(V2CTLSClient client) {
		super();
		tlsClient = client;
	}

	/**
	 * verify certificate
	 */
	public void notifyServerCertificate(TlsServerCertificate serverCertificate) throws IOException {
		try {
			if (keyStore == null) {
				try {
					keyStore = loadKeyStore();
				} catch (Exception e) {
					throw new CertificateException("KeyStore loading failed.");
				}
			}
			KeyStore ks = keyStore;

			CertificateFactory cf = CertificateFactory.getInstance("X.509");
			List<java.security.cert.Certificate> certs = new LinkedList<java.security.cert.Certificate>();
			boolean trustedCertificate = false;
			for ( TlsCertificate c : serverCertificate.getCertificate().getCertificateList()) {
				java.security.cert.Certificate cert = cf.generateCertificate(new ByteArrayInputStream(c.getEncoded()));
				certs.add(cert);

				if (cert instanceof java.security.cert.X509Certificate) {
					java.security.cert.X509Certificate x509cert = (java.security.cert.X509Certificate) cert;
					String subjectDN = x509cert.getSubjectDN().getName();
					String issuerDN = x509cert.getIssuerDN().getName();

					try {
						x509cert.checkValidity();
					}
					catch(CertificateExpiredException e){
						String s = "CertificateExpiredException" + " on " + this.tlsClient.getHost() + "\n";
						String[] arrX509certSystem = (""+x509cert.toString()).split("Certificate\\sExtensions", 2);
						Matcher matcher = null;
						String sVersion = "";
						String sSubject = "";
						String sValidity = "";
						String sIssuer = "";
						matcher = patternVersion.matcher(arrX509certSystem[0]);
						if (matcher.matches()) {
							sVersion = "Version: " + matcher.group(1);
							s += "  " + sVersion + "\n";
						}
						matcher = patternSubject.matcher(arrX509certSystem[0]);
						if (matcher.matches()) {
							sSubject = "Subject: " + matcher.group(1);
							s += "  " + sSubject + "\n";
						}

						if(hmCertificateExpiredException == null){
							hmCertificateExpiredException = new HashMap();
						}
						String key = this.tlsClient.getHost() + "->" + sSubject;
						if(!hmCertificateExpiredException.containsKey(key)){
							matcher = patternValidity.matcher(arrX509certSystem[0]);
							if (matcher.matches()) {
								sValidity = "Validity: " + matcher.group(1).replaceAll("[\n\r]+\\s+", " ");
								s += "  " + sValidity + "\n";
							}
							matcher = patternIssuer.matcher(arrX509certSystem[0]);
							if (matcher.matches()) {
								sIssuer = "Issuer: " + matcher.group(1);
								s += "  " + sIssuer + "\n";
							}
							hmCertificateExpiredException.put(key, sValidity);
							System.out.println(s);
						}						
						continue;
					}
					catch (Exception e) {
						throw e;
					}

					if (subjectDN.equals(issuerDN)) {
						// self signed cert
						try {
							x509cert.verify(x509cert.getPublicKey());
						} catch (Exception e) {
							System.out.println("Self signed certificate verification failed.:"+x509cert.getSubjectDN());
							throw e;
						}
						// self signed certificate is OK.
						if (this.tlsClient.getTlsSocket().isSelfSignPass()) {
							trustedCertificate = true;
						}
						else {
							System.out.println("Self signed certificate cannot pass.:"+x509cert.getSubjectDN());
						}
					}
					else {
						// NOT self signed
						Enumeration en = ks.aliases();
						String alias = "";
						java.security.cert.X509Certificate signCert = null;

						while (en.hasMoreElements()) {
							java.security.cert.X509Certificate storeCert = null;
							alias = (String) en.nextElement();

							if (ks.isCertificateEntry(alias)) {
								storeCert = (java.security.cert.X509Certificate) ks.getCertificate(alias);
								if (storeCert.getIssuerDN().getName().equals(issuerDN)) {
									try {
										x509cert.verify(storeCert.getPublicKey());
										signCert = storeCert;
										break;
									} catch (Exception e) {
										System.out.println("X509 keystore certificate verification failed.:"+storeCert.getSubjectDN());
									}

								}
							}
						}
						if (signCert != null) {
							trustedCertificate = true;
						}
					}
				}

			}
			if (!trustedCertificate) {
				// error
				System.out.println("Not trusted certificate detected.");
				throw new CertificateException("Not trusted certificate detected.:"+this.tlsClient.getHost());
			}

			V2CTLSSession tlsSession = (V2CTLSSession) this.tlsClient.getTlsSocket().getSession();
			tlsSession.setPeerCertArray(certs.toArray(new java.security.cert.Certificate[0]));

		} catch (Exception ex) {
			ex.printStackTrace();
			throw new IOException(ex);
		}
	}

	public TlsCredentials getClientCredentials(CertificateRequest certificateRequest) throws IOException {
		return null;
	}

	private KeyStore loadKeyStore() throws Exception {
		FileInputStream trustStoreFis = null;
		try {
			// Load the JDK's cacerts keystore file or the User's cacerts keystore file
			String filename = System.getProperty("java.home") + "/lib/security/cacerts".replace('/', File.separatorChar);
			// Add for B11 Start
			String Userfilename = (V2CLauncher.getJarDir().getAbsolutePath() + "/cacerts").replace('/', File.separatorChar);
			if((new File(Userfilename).exists())){
				filename = Userfilename;
				System.out.println("BouncyCastle use User's cacerts keystore file => " + filename);
			}
			// Add for B11 END
			trustStoreFis = new FileInputStream(filename);

			KeyStore localKeyStore = null;

			String trustStoreType = System.getProperty("javax.net.ssl.trustStoreType")!=null?System.getProperty("javax.net.ssl.trustStoreType"):KeyStore.getDefaultType();
			String trustStoreProvider = System.getProperty("javax.net.ssl.trustStoreProvider")!=null?System.getProperty("javax.net.ssl.trustStoreProvider"):"";

			if (trustStoreType.length() != 0) {
				if (trustStoreProvider.length() == 0) {
					localKeyStore = KeyStore.getInstance(trustStoreType);
				} else {
					localKeyStore = KeyStore.getInstance(trustStoreType, trustStoreProvider);
				}

				char[] keyStorePass = null;
				String str5 = System.getProperty("javax.net.ssl.trustStorePassword")!=null?System.getProperty("javax.net.ssl.trustStorePassword"):"";
				if (str5.length() <= 0) {
					str5 = "changeit";
				}
				if (str5.length() != 0) {
					keyStorePass = str5.toCharArray();
				}

				localKeyStore.load(trustStoreFis, keyStorePass);
				/*
				Enumeration enumeration = localKeyStore.aliases();
				while(enumeration.hasMoreElements()) {
					String alias = (String)enumeration.nextElement();
					System.out.println("alias name: " + alias);
					java.security.cert.Certificate certificate = localKeyStore.getCertificate(alias);
					System.out.println(certificate.toString());
				}
				*/
				if (keyStorePass != null) {
					for (int i = 0; i < keyStorePass.length; i++) {
						keyStorePass[i] = 0;
					}
				}
			}
			return localKeyStore;
		} finally {
			if (trustStoreFis != null) {
				trustStoreFis.close();
			}
		}
	}
}
